/*************************************************************************
**************************************************************************
Fonction permettant de construire les courbes du livre, notamment pour le chapitre 4 (Modélisation des variables quantitatives)

nb : ce programme inclut en option le tracé de la relation linéaire entre X et Y

dernière mise à jour 8 mars 2025

*************************************************************************
*************************************************************************/



/************************************************************************* 
Ce programme est appelé par la commande logit_crb après un modèle de régression logistique avec une seule variable X. Elle peut être suivi de la commande graph export "fig XXX.png" si on veut enregistrer la figure au format png.

La syntaxe est logit.crb varlist,min(#) max(#) lin(#) obs(#) ic(#) crb(#) nb(#)

- varlist doit contenir les 3 variables : Y (en 0/1), X, X' (X' est la variable X en classes pour lesquelles on veut marquer les points observés
par exemple, pour le fichier geu : Y=ct, X=age, X'=agea)

- min et max sont les valeurs extrêmes de y=Logit P pour les graphes. A choisir empiriquement selon les exemples. Les valeurs par défaut sont -6 et 2.

- lin : 0 pas de tracé du modèle linéaire entre X et Y (valeur par défaut), 1 tracé du modèle linéaire

- obs : 0 observations non figurées,  1 observations présentes (par défaut)

- crb : 0 pas de courbe ; 1 : avec courbe (par défaut). Cela peut être utile si on ne veut que les points observés

- ic : 0 pas d'IC ; 1 : avec IC (par défaut). Utile si on veut "simplifier" la figure.

- nb : 0 si en couleur ; 1 si noir et blanc

*************************************************************************/

/****************************************************
*****************************************************
Exemple (avec les variables du fichier geu)
Courbe sans le modèle linéaire mais avec les valeurs observées et l'IC (ic() n'est pas dans la commande, c'est donc la valeur par défait ic(1) qui est prise)

logit_crb acc age agea ,min(-6) max(3) lin(0) obs(1)

*****************************************************
*****************************************************/


capt program drop logit_crb

*program define logit_crb

program logit_crb

version 11


syntax varlist(min=3 max=3) [if] [in] [ , min(real -6) max(real 2) lin(int 0) obs(int 1) crb(int 1) ic(int 1) nb(int 0)] 

quiet {   // pour que les résultats intermédiaires ne s'affichent
***** 1. Choix pour les graphiques
* scheme
set scheme stcolor

if `nb'==0 {
	* couleur des points observés
	local pcol="sienna"  
	* courbe de la courbe modélisée
	local cc="forest_green"  // couleur courbe
	local colic="gs12"   // couleur de l'intervalle de confiance
	local cw="medthick"  // épaisseur courbe
	local linc="red"  // couleur de la droite
	local linw="medium"  // épaisseur droite
}
else if `nb'==1 {
	* couleur des points observés
	local pcol= "gs8"
	* courbe de la courbe modélisée
	local cc="gs0"  //couleur courbe
	local colic="gs14"   // couleur de l'intervalle de confiance
	local cw="medthick"  // épaisseur courbe
	local linc="gs2"  // couleur de la droite
	local linpat="dashed"
	local linw="medium"  // épaisseur droite
}




***** 2. Récupération dans des macros des paramètres de la commande
tokenize `varlist'
local vary `1'
local varx `2'
local varxa `3'
local legendex : var label `varx'  // la légende de l'axe des abscisses sera le label de la variable X


***** 3. Calculs des bornes de l'intervalle de confiance de la courbe estimée
tempvar etahat se low high
predict `etahat', xb
label var `etahat' "valeur prédite" 
predict `se', stdp
gen `low'=`etahat'-1.96*`se'
gen `high'=`etahat'+1.96*`se'

* Pour qu'on se rende compte si le tracé de l'IC est tronqué si c'est le cas
replace `low'=`min'-1 if `low'<`min'-1                
replace `high'=`max'+1 if `high'>`max'+1 & `high'!=.  

label var `low' "Borne inférieure"
label var `high' "Borne supérieure"


***** 4. Modèle linéaire entre logit P et X (pour le tracer éventuellement)
tempvar linear
logit `vary' `varx'
predict `linear',xb
label var `linear' "modèle linéaire"

***** 5. Calculs des valeurs observées de logit P et marquage des valeurs non calculables (pour P=0 ou P=1) et de leur nombre
tempvar yobs yobs1 yobs2 ycount
bys `varxa': egen `yobs'=mean(`vary')
gen `yobs1'=`min'-0.5 if `yobs'==0
gen `yobs2'= `max'+0.5 if `yobs'==1
replace `yobs'=ln(`yobs'/(1-`yobs')) 
label var `yobs' "Logit observés"
bys `varxa': egen `ycount'=count(`vary')

***** 6. Prise en compte des options de la courbe (observations, IC, ...)
if `obs'==0 {
	local pcol="none"
	local order2
}
else if `obs'==1 {
	local lobs="D"
	local order2 2
}

if `lin'==0 {
	local droite ""
	local order6 
}
else if `lin'==1 {
	local droite "`linear'"
	local order6 6
}

if `crb'==0 {
	local cc "none"
	local order5
}
else if `crb'==1 {
	local order5 5
}

if `ic'==0 {
	local colic "none"
}
else if `ic'==1 {
	*local colic "gs12"
	local order1 1
}

}  // fin de quiet


******* 7. Tracé de la figure
twoway (rarea `low' `high' `varx', sort bcolor(`colic') blcolor(`colic'))  ///
(scatter `yobs' `yobs1' `yobs2' `varxa', sort msymbol(O `lobs' `lobs')  ///
msize(1 .5 .5)  mlabel(. `ycount' `ycount') mlabpos(12 12 12)  ///
mcol(`pcol' `pcol' `pcol') mlabcol(`pcol' `pcol' `pcol'))  ///
(line `etahat' `droite' `varx', sort lpattern(solid dash)  ///
lwidth(`cw' `linw') lcolor(`cc' `linc'))  ///
, ylabel(`min' (2) `max',nogrid) xlabel(, nogrid) l1title("Logit P")  xtitle("`legendex'")  ///
legend (cols(2) order(`order5' `order1' `order6' `order2') label(1 "Intervalle de confiance")  pos(11) ring(0) size(small) symxsize(*.5) symysize(*.5) region(lcol(black)))


end	
	

